/****************************************************************************************/
/**
  Copyright (c) 2008 Freescale Semiconductor
  Freescale Confidential Proprietary
  \file     	Pwm.c
  \brief    	This file Initialize all PIT Hw resources to emulate Pwm signals   \n
                and handles all initialization parameters to operate correctly the \n
                emulated Pwm module.
  \author   	Freescale Semiconductor
  \author       r01160
  \author   	Guadalajara Applications Laboratory RTAC Americas
  \version      0.1
  \date     	4/16/2008 9:51:45 AM
  \warning      
  
  * History:
  
*/
/****************************************************************************************/

/** Pwm function declarations */
#include "Pwm.h"

#if (DIAGNOSTICS == ON)   /* Check if Diagnostics is needed */
  #include "Adc.h"
#endif  

/** This pointer shall access to Pwm configuration structure */
Pwm_ChannelConfigType* current_chp = NULL_PTR;
/*---------------------------- Global variables -------------------------*/
/** External Interrupt table */
extern const uint32_t IntcIsrVectorTable_p1[];
/** Holds the last state of the Pwm module */
const Pwm_ChannelConfigType *Pwm_Cfg_Ptr = NULL_PTR;

/**
 * \brief     Initialize all Pwm resources to operate correctly. \n
              Intialization includes GPIO setup, Pit timer setup \n
              to generate the desired Pwm period and Pwm parameters.
 * \author    R01160
 * \param     const Pwm_ChannelConfigType *PortCfgPtr
 * \return    none
 * \todo
 * \warning   Only pins contained in Pwm_ChannelConfig structure will be configured. \n
              if Report Error Layer (PWM_ERROR_DETECT == OFF) is turned off, none Pwm \n
              parameters shall be analyzed and possible corrupt data might be generated.
 */  
 void vfnPwm_Init(const Pwm_ChannelConfigType *PwmCfgPtr)
 {
    #if (PWM_ERROR_DETECT == ON)   /* If Pwm Error Layer Enabled, validate errors */
      if(PwmCfgPtr == NULL_PTR)   /* Missing Configuration Pointer?              */
      { 
        vfnPwm_Report_Error(PWM_INIT_ID,PWM_E_PARAM_CONFIG); 
      }
      else
      {
    #endif 
    
    /* First Initialize all GPIO pins to operate as Pwm Channels        */
     vfnPort_Init(PwmCfgPtr);
     
   #if (DIAGNOSTICS == ON)   /* Check if Diagnostics is needed */
	   /* Initialize DMA engine to handle ADC conversions */
	   vfnedma_init();
	   /* Configures TCD channels 0 & 1                   */
	   vfndma_config_fcn();
	   /* Initialize ADC0 module with predefined parameters from Adc_Cfg.h file */
	   vfnAdc_Init();
	   /* Start ADC0 Calibration  */
	   vfnInitAdc_Calibration(ADC_FIFO_BUFFER);
   #endif
     /* Initialize PIT # timer to have a resolution of 0.5% period value */
     vfnPit_Init(PIT_CH_1, (uint32_t)SET_33US_TIMEBASE); 
    /* Pwm module already initialized */
    Pwm_Cfg_Ptr = PwmCfgPtr;
    
    #if (PWM_ERROR_DETECT == ON)   /* If Pwm Error Layer Enabled, validate errors */
      }
    #endif  
    
 }
 
 /*--------------------------------------------------------------------------------------*/ 
 /**
 * \brief     Configure HVEN register to let MCU know which Core will handle Pit Interrupts.
 * \author    R01160
 * \param     Interrupt Type (Hw or Sw)
 * \return    none
 * \todo      
 * \warning   Only two choices are allowed to operate within the specification on this \n
              function. Z1 core OR Z0 core (Z1 or Z0 values). 
 */
 void vfnCoreInterruptHandling(uint8_t u8InterruptType)
 {	
	 #if(Z0_CORE == ON)                      /* Z0 core will handle Pit interrupts?              */
	 	  INTC_HVEN_PRC1 = u8InterruptType;  /* MPC551x: Proc'r 0: initialize for HW vector mode */	
	 	  INTC.IACKR_PRC1.R = (uint32_t) &IntcIsrVectorTable_p1[0]; /* MPC551x: ISR table base */
	 #else
	 	 INTC_HVEN_PCR0 = u8InterruptType;   /* MPC551x: Proc'r 1: initialize for HW/SW vector mode */	
	 #endif
 	
 }
 
 /*--------------------------------------------------------------------------------------*/
/**
 * \brief     Change PWM Period and Duty Cycle during runtime. Period and Duty Cycle \n
              shall be refresed once the last period ends.
 * \author    R01160
 * \param     Pwm channel,Pwm Period , Duty Cycle value
 * \return    none
 * \todo
 * \warning   User must be aware to pass only valid configured Pwm Channels.
 */  
 void vfnSetPeriodAndDuty(uint8_t u8PwmChannel, uint32_t u32Period ,uint32_t u32DutyCycle)
 {
     
    #if (PWM_ERROR_DETECT == ON)          /* If Pwm Error Layer Enabled validate errors */	
      if(Pwm_Cfg_Ptr == NULL_PTR)        /* Pwm module already initilized?             */     
      {
         vfnPwm_Report_Error(PWM_SETPERIODANDDUTY_ID,PWM_E_UNINIT); /* Report None initilized module */
      }
      else if (u32DutyCycle > DUTY_CYCLE_100)
      {
         vfnPwm_Report_Error(PWM_SETPERIODANDDUTY_ID,PWM_E_DUTYCYCLE_RANGE);        
      } 
      else if(u32DutyCycle > u32Period)  /* Check if Duty Cycle is defined within the range of Period */
      {
      	vfnPwm_Report_Error(PWM_SETPERIODANDDUTY_ID,PWM_E_PERIOD_DUTY_RANGE);        
      }
      else
      {
   #endif 
       
         current_chp = &Pwm_ChannelConfig[u8PwmChannel]; 
 		 current_chp->u32Period = (uint32_t)u32Period;
	   	 current_chp->u32Duty   = (uint32_t)u32DutyCycle; /* Set New Duty cycle    */
	
   #if (PWM_ERROR_DETECT == ON)   /* If Pwm Error Layer Enabled, validate errors */
      }
   #endif 
    	
 }
 
 
/*--------------------------------------------------------------------------------------*/
/**
 * \brief     Change PWM Duty Cycle during runtime.
 * \author    R01160
 * \param     Pwm channel, Duty Cycle value
 * \return    none
 * \todo
 * \warning   User must be aware to pass only valid configured Pwm Channels.\n
 *            PWM Channel values MUST START from 0-up to configured Pwm     \n
 *            channels. i.e. Pwm_Chan_0 = 0,  Pwm_Chan_1 = 1, etc. 
 */  
 void vfnSetDuty_Cycle(uint8_t u8PwmChannel, uint32_t u32DutyCyle)
 {
     
    #if (PWM_ERROR_DETECT == ON)          /* If Pwm Error Layer Enabled validate errors */	
      if(Pwm_Cfg_Ptr == NULL_PTR)        /* Pwm module already initilized?             */     
      {
         vfnPwm_Report_Error(PWM_SETDUTYCYCLE_ID,PWM_E_UNINIT); /* Report None initilized module */
      }
      else if (u32DutyCyle > MAX_DUTY_CYCLE)
      {
         vfnPwm_Report_Error(PWM_SETDUTYCYCLE_ID,PWM_E_DUTYCYCLE_RANGE);        
      }      
      else
      {
   #endif 
   
   /* Point to Pwm channel to change the new requested duty cycle value */
   current_chp = &Pwm_ChannelConfig[u8PwmChannel]; 
   /* Set New Duty cycle. New Duty Cycle value shall be changed when the on-going Period finishes  */
   current_chp->u32Duty = (uint32_t)u32DutyCyle; 
    	  
 	
   #if (PWM_ERROR_DETECT == ON)   /* If Pwm Error Layer Enabled, validate errors */
      }
   #endif 
    	
 }
 
 /*--------------------------------------------------------------------------------------*/
/**
 * \brief     Generate signals for all Pwm channels properly configured in Pwm_Cfg.h file.
 * \author    R01160
 * \param     none
 * \return    none
 */
/*--------------------------------------------------------------------------------------*/
 asm void Pwm_Generation_Asm_Fnc(void)
{              
                 lis    r8,Pwm_ChannelConfig@ha
	             addi r7,r8,Pwm_ChannelConfig@l
	             li    r0,PWM_MAX_CHANNELS
	             mtspr    9,r0         /* Load Total Pwm channels into Counter register        */
    NextChannel: lwz   r4,4(r7)        /* Port Channel                                         */
			     lwz   r5,8(r7)        /* Load  Period value                                   */
			     lwz   r6,12(r7)       /* Load Duty Cycle                                      */
			     lwz   r0,16(r7)       /* Load Channel Counter                                 */
			     cmpi  r0,0            /* Is period Over ?                                     */
			     bne   PeriodNotOver
			     lwz   r0,8(r7)        /* reload new period info                               */
			     stw   r0,16(r7)       /* Update Channel Cntr                                  */
			  #if (DIAGNOSTICS == ON)  /* Diagnostics Enabled ?                                */
			     lwz      r3,24(r7)    /* Load Adc trigger value                               */
			     cmp      r0,r3        /* Validate if Cntr value has reached Adc trigger value */
			     bne      ExitTrigger  /* Jump if trigger value not yet reached                */
			     lwz      r3,20(r7)    /* Load ADC Channel to be sampled                       */         /* Saves vfnStart_Conversion_fnc params (adc channel)   */  
                 se_extzb r3
			     se_bl     vfnStart_Conversion_fnc  /* Start Adc Conversion                     */
              #endif
    ExitTrigger: lis    r8,SIU_GPDO_BASE_ADDR@ha
                 addi r8,r8,SIU_GPDO_BASE_ADDR@l
                 add   r8,r8,r4        /* Assembling Port address                              */
                 cmp   r0,r6           /* Duty has been reached?                               */
			     bne   TurnSignalOff   /* turn off channel if duty cycle not reached yet       */
   TurnSignalOn: li    r0,1        
                 e_stb r0,0(r8)        /* Turn signal on                                       */
                 e_b  GotoNextCh      /* Jump to next PWM channel                             */
  TurnSignalOff: li    r0,0
                 e_stb r0,0(r8)        /* Turn pin signal OFF                                  */
                 se_b  GotoNextCh      /* Jump to next PWM channel                             */
  PeriodNotOver: cmp r0,r6
                 bne GotoNextCh        /* Jump to next PWM channel                             */
                 se_b TurnSignalOn     /* Turn pin signal ON                                   */ 
     GotoNextCh: lwz   r0,16(r7)
                 li    r4,1
                 sub   r0,r0,r4
                 stw   r0,16(r7)
              #if (DIAGNOSTICS == ON)  /* Jump 3 more locations if Diagnostics enabled         */   
                 addi r7,r7,32
              #else
                 addi r7,r7,20
              #endif   
			     bdnz  NextChannel     /* All Pwm channels have been analyzed?                 */
                 
}
 
/*--------------------------------------------------------------------------------------*/
/**
 * \brief     Generate signals for all Pwm channels properly configured in Pwm_Cfg.h file.
 * \author    R01160
 * \param     none
 * \return    none
 */
/*--------------------------------------------------------------------------------------*/
 void Pwm_Generation_Fnc(void)
 {
    current_chp = &Pwm_ChannelConfig[PWM_MAX_CHANNELS];         /* Load new PWM configuration */
   
  	  /* violation of MISRA 2004 rule 17.3 */
  	  while (current_chp>=&Pwm_ChannelConfig[0])               /* for all channels... */
	  {	
	    if(current_chp->u32Cntr==(uint32_t)0)                  /* if period is over... */
	    {
	        current_chp->u32Cntr = current_chp->u32Period;	   /* Reload new period information */
	     
	      #if (DIAGNOSTICS == ON)                              /* Diagnostics Enabled ?         */
	        if(current_chp->u32Cntr==current_chp->u32Adc_Trigger_Value) /* Triggers and Adc conversion if value reaches Pwm Cntr */
	        {
	           vfnStart_Conversion_fnc(ADC_CHANNEL_0,ADC_CAL_ENABLE);   /* Start Conversion using channel 0 */
	           current_chp->u32Adc_Fifo_Buffer = (uint16_t)u16Read_Adc_Result(ADC_FIFO_BUFFER); /* Save Adc result*/
	        }
	     #endif
	        
	        if(current_chp->u32Cntr==current_chp->u32Duty)     /* if duty (switch on point) has been reached (to achieve real 100% duty cycles) */
	        {
	          SET_PIN_HIGH(current_chp->u8Pwm_Port_Channel);   /* switch channel on by write a one to the corresponding pin */
	        }    
	   		else
	        {
	          SET_PIN_LOW(current_chp->u8Pwm_Port_Channel);      /* ...switch channel off               */
	        }  
	    } 
	    else 
	    {
	        if(current_chp->u32Cntr==current_chp->u32Duty)     /* if duty (switch on point) has been reached */
	        {
	          SET_PIN_HIGH(current_chp->u8Pwm_Port_Channel);   /* switch channel on by write a one to the corresponding pin */
	        }
	    }
	    current_chp->u32Cntr--;          	                   /* increase counter which represents position within period for each channel */
	    current_chp--;  					                   /* next channel please                                           */
	  }
 }
